-- Echoes of the Unsaid v1.0
-- A Minetest mod that listens to silence, hesitation, and ritual pauses.

local modname = minetest.get_current_modname()
local S = minetest.get_translator(modname)

-- =========================
-- Player state tracking
-- =========================

local players = {}

local function init_player(name)
    players[name] = {
        last_pos = nil,
        still_ticks = 0,       -- short-term current pause
        silence_score = 0,     -- long-term attunement
        last_echo_time = 0,    -- last time an echo event happened
        memory_entries = {},   -- { {pos=, env=, tone=} }
        ritual_flags = {},
    }
end

minetest.register_on_joinplayer(function(player)
    local name = player:get_player_name()
    init_player(name)
    minetest.chat_send_player(name, S("You feel the world pause with you for a moment."))
end)

minetest.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    players[name] = nil
end)

-- =========================
-- Utility helpers
-- =========================

local function pos_equal(a, b)
    if not a or not b then return false end
    local dx = a.x - b.x
    local dy = a.y - b.y
    local dz = a.z - b.z
    return (dx * dx + dy * dy + dz * dz) < 0.01
end

local function env_tag_from_node(node_name, pos)
    if not node_name then return "unknown" end
    if node_name:find("water") then
        return "water"
    elseif node_name:find("lava") then
        return "danger"
    elseif node_name:find("stone") or node_name:find("cobble") then
        if pos.y < -20 then
            return "underground"
        else
            return "rocky"
        end
    elseif pos.y > 40 then
        return "height"
    elseif node_name:find("sand") then
        return "shore"
    elseif node_name:find("tree") or node_name:find("leaves") then
        return "forest"
    end
    return "plain"
end

local function add_memory_entry(state, pos, env, tone)
    local entry = { pos = vector.round(pos), env = env, tone = tone or "neutral" }
    table.insert(state.memory_entries, entry)
    if #state.memory_entries > 10 then
        table.remove(state.memory_entries, 1)
    end
end

-- =========================
-- Echo node
-- =========================
-- Echo Node (updated to drop Echo Dust)
minetest.register_node(modname .. ":echo_node", {
    description = S("Faint Echo"),
    drawtype = "glasslike",
    tiles = { "echoes_of_the_unsaid_echo.png" },
    use_texture_alpha = "blend",
    sunlight_propagates = true,
    paramtype = "light",
    light_source = 3,
    groups = { oddly_breakable_by_hand = 3, not_in_creative_inventory = 1 },
    drop = modname .. ":echo_dust",  -- Echo Dust now drops naturally

    on_destruct = function(pos)
        local players_near = minetest.get_connected_players()
        for _, p in ipairs(players_near) do
            local pp = p:get_pos()
            if vector.distance(pp, pos) < 8 then
                local name = p:get_player_name()
                minetest.chat_send_player(name, S("Something you never said fractures quietly."))
            end
        end
    end,
})


-- Fallback texture note:
-- If you don't add echoes_of_the_unsaid_echo.png,
-- it will appear as an unknown node texture; still functional.

-- =========================
-- Attunement + Echo logic
-- =========================

local TICK_INTERVAL = 1.0         -- seconds between checks
local STILL_THRESHOLD = 3         -- seconds before we start counting as still
local ECHO_COOLDOWN = 20          -- seconds between possible echo events
local MAX_SILENCE_SCORE = 1000

local timer_accum = 0

local function get_attunement_tier(score)
    if score < 50 then
        return 1
    elseif score < 150 then
        return 2
    elseif score < 400 then
        return 3
    elseif score < 800 then
        return 4
    else
        return 5
    end
end

local function maybe_trigger_echo(player, state)
    local now = minetest.get_gametime()
    if now - state.last_echo_time < ECHO_COOLDOWN then
        return
    end

    local pos = player:get_pos()
    if not pos then return end
    local name = player:get_player_name()

    -- Slightly above/below for underfoot
    local under = { x = pos.x, y = pos.y - 1, z = pos.z }
    local node_under = minetest.get_node_or_nil(under)
    if not node_under then return end

    local env = env_tag_from_node(node_under.name, under)

    local tier = get_attunement_tier(state.silence_score)
    local base_chance = 0.05
    local tier_bonus = (tier - 1) * 0.05
    local chance = base_chance + tier_bonus

    -- Additional weighting for rituals (simple flags; can be expanded)
    if state.ritual_flags.threshold_hesitation then
        chance = chance + 0.05
    end
    if state.ritual_flags.water_attuned then
        chance = chance + 0.05
    end

    if math.random() > chance then
        return
    end

    state.last_echo_time = now

    -- Add a memory entry for this pause
    local tone = "lingering"
    if env == "height" then
        tone = "edge"
    elseif env == "water" then
        tone = "dissolved"
    end
    add_memory_entry(state, pos, env, tone)

    -- Choose a type of echo event
    local r = math.random()

    if r < 0.4 then
        -- Subtle chat whisper
        local lines = {
            "Something waits for you to move again.",
            "The ground beneath you feels strangely attentive.",
            "Your stillness hums like a held breath.",
            "This is not the first time you paused here.",
        }
        local msg = lines[math.random(#lines)]
        minetest.chat_send_player(name, S(msg))

    elseif r < 0.7 then
        -- Create an echo node nearby if safe
        local offset_choices = {
            {x=0, y=-1, z=0},
            {x=1, y=0, z=0},
            {x=-1, y=0, z=0},
            {x=0, y=0, z=1},
            {x=0, y=0, z=-1},
        }
        local off = offset_choices[math.random(#offset_choices)]
        local target = vector.add(vector.round(pos), off)
        local target_node = minetest.get_node_or_nil(target)
        if target_node then
            local def = minetest.registered_nodes[target_node.name]
            local safe = true
            if def then
                if def.groups and (def.groups.ore or def.groups.treasure) then
                    safe = false
                end
                if def.protected or def.name == "default:chest" then
                    safe = false
                end
            end
            if safe then
                minetest.set_node(target, { name = modname .. ":echo_node" })
            end
        end

        minetest.chat_send_player(name, S("Something in the world rearranges itself around your pause."))

    else
        -- Time-lagged callback referencing where they were
        local past_pos = vector.round(pos)
        minetest.after(7 + math.random(10), function()
            local p = minetest.get_player_by_name(name)
            if not p then return end
            local current = p:get_pos()
            if not current then return end
            local dist = vector.distance(current, past_pos)
            if dist > 2 then
                minetest.chat_send_player(name, S(
    "You moved on. That moment stayed behind at @1, @2, @3.",
    past_pos.x, past_pos.y, past_pos.z
))

            else
                minetest.chat_send_player(name, S("You’re still here. The moment thickens around you."))
            end
        end)
    end
end

-- =========================
-- Ritual heuristics
-- =========================

local function update_rituals(player, state, dt)
    local pos = player:get_pos()
    if not pos then return end

    local name = player:get_player_name()
    local under = { x = pos.x, y = pos.y - 1, z = pos.z }
    local node_under = minetest.get_node_or_nil(under)
    local under_name = node_under and node_under.name or ""
    local env = env_tag_from_node(under_name, under)

    -- Water ritual: long stillness in water increases affinity
    if env == "water" and state.still_ticks * TICK_INTERVAL > 10 then
        state.ritual_flags.water_attuned = true
        -- Soft one-time hint
        if not state.ritual_flags._water_hint_sent then
            state.ritual_flags._water_hint_sent = true
            minetest.chat_send_player(name, S("The water feels like it’s listening."))
        end
    end

    -- Threshold hesitation: near edges
    if env == "height" or env == "shore" then
        -- Rough heuristic: if y is high or near drop and still
        if state.still_ticks * TICK_INTERVAL > 5 then
            state.ritual_flags.threshold_hesitation = true
        end
    end

    -- You can expand rituals here later if desired.
end

-- =========================
-- Globalstep: silence tracking
-- =========================

minetest.register_globalstep(function(dtime)
    timer_accum = timer_accum + dtime
    if timer_accum < TICK_INTERVAL then
        return
    end
    local dt = timer_accum
    timer_accum = 0

    for _, player in ipairs(minetest.get_connected_players()) do
        local name = player:get_player_name()
        local state = players[name]
        if not state then
            init_player(name)
            state = players[name]
        end

        local pos = player:get_pos()
        if not pos then
            goto continue
        end

        if state.last_pos and pos_equal(vector.round(pos), vector.round(state.last_pos)) then
            state.still_ticks = state.still_ticks + 1
        else
            -- If they were previously still, reward silence_score a bit
            if state.still_ticks * TICK_INTERVAL >= STILL_THRESHOLD then
                local added = math.min(state.still_ticks, 30)
                state.silence_score = math.min(MAX_SILENCE_SCORE, state.silence_score + added)
            end
            state.still_ticks = 0
            state.last_pos = vector.round(pos)
        end

        -- If currently in a still state, maybe trigger things
        if state.still_ticks * TICK_INTERVAL >= STILL_THRESHOLD then
            update_rituals(player, state, dt)
            maybe_trigger_echo(player, state)
        end

        ::continue::
    end
end)

-- =========================
-- /echo_state command
-- =========================

minetest.register_chatcommand("echo_state", {
    description = S("Sense how attuned you are to the quiet."),
    func = function(name)
        local state = players[name]
        if not state then
            return false, S("The world doesn’t seem to recognize you yet.")
        end

        local tier = get_attunement_tier(state.silence_score)
        local msg

        if tier == 1 then
            msg = "You feel like any other traveler. The world isn’t watching yet."
        elseif tier == 2 then
            msg = "Something occasionally pricks at the edge of your awareness."
        elseif tier == 3 then
            msg = "You suspect pauses mean more here than they should."
        elseif tier == 4 then
            msg = "This world holds onto moments when you stood still."
        else
            msg = "You are threaded into the quiet between seconds."
        end

        local mem_count = #state.memory_entries
        local extra = ""
        if mem_count > 0 then
            extra = " You’ve left " .. mem_count .. " unspoken places behind you."
        end

        return true, S(msg .. extra)
    end,
})

local echo_biomes = {}

local function activate_echo_biome(pos)
    local key = minetest.pos_to_string(vector.round(pos))
    echo_biomes[key] = {
        activated_at = minetest.get_gametime(),
        pos = vector.round(pos),
    }
end

local function is_echo_biome(pos)
    local key = minetest.pos_to_string(vector.round(pos))
    local entry = echo_biomes[key]
    if not entry then return false end
    local age = minetest.get_gametime() - entry.activated_at
    return age < 600  -- 10 minutes
end

minetest.register_entity(modname .. ":whisperling", {
    initial_properties = {
        physical = false,
        collide_with_objects = false,
        visual = "sprite",
        textures = { "echoes_of_the_unsaid_whisperling.png" },
        visual_size = { x = 0.5, y = 0.5 },
        pointable = false,
        glow = 5,
    },
    on_step = function(self, dtime)
        local pos = self.object:get_pos()
        if not pos then return end
        local players = minetest.get_connected_players()
        for _, p in ipairs(players) do
            local pp = p:get_pos()
            if vector.distance(pp, pos) < 4 then
                if math.random() < 0.01 then
                    minetest.chat_send_player(p:get_player_name(), "A whisperling drifts near your silence.")
                end
            end
        end
    end,
})


local function maybe_spawn_whisperling(player, state)
    if math.random() > 0.02 then return end
    local pos = player:get_pos()
    if not pos then return end
    local nearby = minetest.get_objects_inside_radius(pos, 10)
    for _, obj in ipairs(nearby) do
        if obj:get_luaentity() and obj:get_luaentity().name == modname .. ":whisperling" then
            return
        end
    end
    local spawn_pos = vector.add(pos, { x = math.random(-2,2), y = 1, z = math.random(-2,2) })
    minetest.add_entity(spawn_pos, modname .. ":whisperling")
end

local function maybe_add_codex_entry(player, state, env)
    if not state.codex then state.codex = {} end
    if #state.codex >= 20 then return end
    local pos = vector.round(player:get_pos())
    local tone = "reflective"
    local line = "You paused at " .. env .. ". The world wrote it down."
    table.insert(state.codex, { pos = pos, env = env, tone = tone, line = line })
end
minetest.register_chatcommand("spawn_whisperling", {
    description = "Spawn a Whisperling for testing",
    privs = { server = true },
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if not player then
            return false, "Player not found."
        end

        local pos = player:get_pos()
        local spawn_pos = vector.add(pos, { x = 0, y = 1, z = 0 })

        minetest.add_entity(spawn_pos, modname .. ":whisperling")
        return true, "Whisperling spawned."
    end,
})

minetest.register_chatcommand("echo_codex", {
    description = "View your codex of silent places",
    func = function(name)
        local state = players[name]
        if not state or not state.codex then
            return true, "Your codex is empty. The world hasn’t recorded your silences yet."
        end
        local lines = {}
        for i, entry in ipairs(state.codex) do
            table.insert(lines, i .. ". " .. entry.line)
        end
        return true, table.concat(lines, "\n")
    end,
})


minetest.register_craftitem(modname .. ":echo_dust", {
    description = "Echo Dust",
    inventory_image = "echoes_of_the_unsaid_dust.png",
})

minetest.register_craftitem(modname .. ":stillglass", {
    description = "Stillglass",
    inventory_image = "echoes_of_the_unsaid_stillglass.png",
})

minetest.register_craftitem(modname .. ":codex_tablet", {
    description = "Codex Tablet",
    inventory_image = "echoes_of_the_unsaid_tablet.png",
    on_use = function(itemstack, user)
        local name = user:get_player_name()
        minetest.chat_send_player(name, "Use /echo_codex to view your silent places.")
        return itemstack
    end,
})

minetest.register_craft({
    output = modname .. ":stillglass",
    recipe = {
        { "", modname .. ":echo_dust", "" },
        { "", "default:glass", "" },
        { "", "", "" },
    },
})

minetest.register_craft({
    output = modname .. ":codex_tablet",
    recipe = {
        { "", modname .. ":stillglass", "" },
        { "", "default:book", "" },
        { "", "", "" },
    },
})

local function maybe_shimmer_memory_zone(player, state)
    if not state.memory_entries then return end
    local pos = vector.round(player:get_pos())
    for _, entry in ipairs(state.memory_entries) do
        if vector.distance(pos, entry.pos) < 3 then
            if math.random() < 0.05 then
                minetest.chat_send_player(player:get_player_name(), "This place remembers your silence.")
                activate_echo_biome(pos)
            end
        end
    end
end

local function phase2_globalstep(player, state)
    local tier = get_attunement_tier(state.silence_score)
    if tier >= 3 then
        maybe_spawn_whisperling(player, state)
        maybe_add_codex_entry(player, state, "unknown")
        maybe_shimmer_memory_zone(player, state)
    end
end

local old_globalstep = minetest.registered_globalsteps[#minetest.registered_globalsteps]
minetest.register_globalstep(function(dtime)
    if old_globalstep then old_globalstep(dtime) end
    for _, player in ipairs(minetest.get_connected_players()) do
        local name = player:get_player_name()
        local state = players[name]
        if state then
            phase2_globalstep(player, state)
        end
    end
end)

